#pragma once
#define ENABLE_SERIAL true

namespace EEGSampler
{

	using namespace System;
	using namespace System::ComponentModel;
	using namespace System::Collections;
	using namespace System::Windows::Forms;
	using namespace System::Data;
	using namespace System::Drawing;
	using namespace System::Drawing::Printing;  //added manually
	using namespace System::Globalization;      // ditto
	using namespace System::Threading;			// ditto
	using namespace System::IO::Ports;
	using namespace System::IO;

	/// <summary>
	/// Summary for MyForm
	/// </summary>

	public ref class MyForm : public System::Windows::Forms::Form
	{
	public: PrintDocument ^printDocument1 = gcnew PrintDocument();	// added to get form print going
	private: System::Windows::Forms::PictureBox^ pictureBox1;
	private: System::Windows::Forms::ComboBox^  comboBox1;
	private: System::Windows::Forms::TextBox^  textBox1;
	private: System::Windows::Forms::Timer^ myTimer = gcnew System::Windows::Forms::Timer;

	public: array<Int16> ^intSampleArray1 = gcnew array<Int16>(20000);	// array for Ch1 samples
	public: array<Int16> ^intSampleArray2 = gcnew array<Int16>(20000);	// array for Ch2 samples
	public: array<Int16> ^intSampleArray3 = gcnew array<Int16>(20000);	// array for Ch3 samples

	public:
		MyForm(void)
		{
			// Silicon Chip EEG Sampler control program V1.0
			// Written by Jim Rowe.
			// Last revised 27/06/2018 at 9:10 am

			InitializeComponent();
//			findPorts();
		}

	protected:
		/// <summary>
		/// Clean up any resources being used.
		/// </summary>

	private: System::Windows::Forms::MenuStrip^  menuStrip1;
	private: System::Windows::Forms::ToolStripMenuItem^  fileToolStripMenuItem;
	private: System::Windows::Forms::ToolStripMenuItem^  openToolStripMenuItem;
	private: System::Windows::Forms::ToolStripMenuItem^  saveToolStripMenuItem;
	private: System::Windows::Forms::ToolStripMenuItem^  printToolStripMenuItem;
	private: System::Windows::Forms::ToolStripMenuItem^  exitToolStripMenuItem;
	private: System::Windows::Forms::ToolStripMenuItem^  aboutToolStripMenuItem;
	private: System::Windows::Forms::CheckBox^ checkbox1;
	private: System::Windows::Forms::Button^  button1;
	private: System::Windows::Forms::Label^  label1;
	private: System::Windows::Forms::ComboBox^  comboBox2;
	private: System::Windows::Forms::Label^  label2;
	private: System::Windows::Forms::ComboBox^  comboBox3;
	private: System::Windows::Forms::Label^  label3;
	private: System::Windows::Forms::ProgressBar^  progressBar1;
	private: System::Windows::Forms::Label^  label4;
	private: System::IO::Ports::SerialPort^  serialPort1;
	private: System::Windows::Forms::OpenFileDialog^  openFileDialog1;
	private: System::Windows::Forms::SaveFileDialog^  saveFileDialog1;
	private: System::Windows::Forms::PrintDialog^  printDialog1;
	private: System::ComponentModel::IContainer^  components;

	private:
		/// <summary>
		/// Required designer variables
		Int16 intSamplingTime;		// total sampling time in seconds (5, 15, 30 or 60)
		Int16 intTLCntr;			// seconds timeline counter
		Int16 intSecSp;				// spacing for seconds time lines
		Int16 intScaledSmplVal;		// scaled sample value variable (0-160)
		volatile Int16 intCurrArrayCntr;	// counter for current sample in arrays
		Int16 intSmplPerSec = 250;	// number of sample trios taken per second
		Int16 intNumofSamples;		// number of samples to be taken (5, 15, 30 or 60 times 267)
		volatile bool samplingActive;	// flag to show whether sampling is under way
		int samplingLastShownPoint;
		PointF point1;				// point 1 for graph line drawing
		PointF point2;				// point 2 for graph line drawing
		System::Drawing::Graphics ^FormGraphic;		// handle to manage form's drawing activities
		System::Drawing::Pen ^myBlackPen, ^myRedPen;
		System::Drawing::SolidBrush ^myBrush = gcnew System::Drawing::SolidBrush(System::Drawing::Color::White);
		System::Drawing::Point ptStart, ptEnd;	// drawing coordinates

	private: System::Void StopSampling() {
		this->myTimer->Stop();
		this->progressBar1->Value = 1;		// return to the bar's initial value
		this->button1->Text = "Click to Start Sampling";
		this->samplingActive = false;

		// & re-enable the toolstrip menu items before leaving
		this->openToolStripMenuItem->Enabled = true;
		this->saveToolStripMenuItem->Enabled = true;
		this->printToolStripMenuItem->Enabled = true;
		this->exitToolStripMenuItem->Enabled = true;
	}
	private: System::Void TimerEventProcessor(Object^ myObject, EventArgs^ myEventArgs)
	{
		bool timeout = false;
		bool redraw_all = false;
		int i = 0;
#ifdef ENABLE_SERIAL
		while (this->serialPort1->BytesToRead > 0 && !timeout && this->samplingActive && intCurrArrayCntr+i < intNumofSamples) {
			try
			{
				intSampleArray1[intCurrArrayCntr+i] = Int16::Parse(this->serialPort1->ReadLine()); // read 1st line & save in Array1
				intSampleArray2[intCurrArrayCntr+i] = Int16::Parse(this->serialPort1->ReadLine()); // read 2nd line & save in Array2
				intSampleArray3[intCurrArrayCntr+i] = Int16::Parse(this->serialPort1->ReadLine()); // read 3rd line & save in Array3
				++i;
				if (this->checkbox1->Checked && intCurrArrayCntr + i == intNumofSamples) {
					Array::Copy(intSampleArray1, 1, intSampleArray1, 0, intNumofSamples - 1);
					Array::Copy(intSampleArray2, 1, intSampleArray2, 0, intNumofSamples - 1);
					Array::Copy(intSampleArray3, 1, intSampleArray3, 0, intNumofSamples - 1);
					--intCurrArrayCntr;
					redraw_all = true;
				}
			}
			catch (TimeoutException^)
			{
				timeout = true;
			}
		}
#endif
		if (timeout) {
			StopSampling();
#ifdef ENABLE_SERIAL
			this->serialPort1->Close();			// and close port
#endif
			MessageBox::Show("Port not responding", "Timeout exception", MessageBoxButtons::OK);
		}
		else {
			if (redraw_all) {
				this->pictureBox1->Invalidate();
//				RedrawGraticuleandPlots(FormGraphic, Rectangle(0, 0, this->pictureBox1->Bounds.Width, this->pictureBox1->Bounds.Height), 2.0f);
			} else {
				while (i--) {
					UpdatePlots(FormGraphic, Rectangle(0, 0, this->pictureBox1->Bounds.Width, this->pictureBox1->Bounds.Height));
					++intCurrArrayCntr;
				}
			}
			if (!this->samplingActive || intCurrArrayCntr >= intNumofSamples) {
				StopSampling();
#ifdef ENABLE_SERIAL
				this->serialPort1->WriteLine("s");	// send stop sampling command char
				this->serialPort1->Close();			// and close port
#endif
			}
		}
	}
		/// </summary>
// ----------------------------------------------------------------------------------------------------
#pragma region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		void InitializeComponent(void)
		{
			this->components = (gcnew System::ComponentModel::Container());
			this->menuStrip1 = (gcnew System::Windows::Forms::MenuStrip());
			this->fileToolStripMenuItem = (gcnew System::Windows::Forms::ToolStripMenuItem());
			this->openToolStripMenuItem = (gcnew System::Windows::Forms::ToolStripMenuItem());
			this->saveToolStripMenuItem = (gcnew System::Windows::Forms::ToolStripMenuItem());
			this->printToolStripMenuItem = (gcnew System::Windows::Forms::ToolStripMenuItem());
			this->exitToolStripMenuItem = (gcnew System::Windows::Forms::ToolStripMenuItem());
			this->aboutToolStripMenuItem = (gcnew System::Windows::Forms::ToolStripMenuItem());
			this->checkbox1 = (gcnew System::Windows::Forms::CheckBox());
			this->button1 = (gcnew System::Windows::Forms::Button());
			this->label1 = (gcnew System::Windows::Forms::Label());
			this->comboBox2 = (gcnew System::Windows::Forms::ComboBox());
			this->label2 = (gcnew System::Windows::Forms::Label());
			this->comboBox3 = (gcnew System::Windows::Forms::ComboBox());
			this->label3 = (gcnew System::Windows::Forms::Label());
			this->progressBar1 = (gcnew System::Windows::Forms::ProgressBar());
			this->label4 = (gcnew System::Windows::Forms::Label());
			this->pictureBox1 = (gcnew System::Windows::Forms::PictureBox());
			this->serialPort1 = (gcnew System::IO::Ports::SerialPort(this->components));
			this->openFileDialog1 = (gcnew System::Windows::Forms::OpenFileDialog());
			this->saveFileDialog1 = (gcnew System::Windows::Forms::SaveFileDialog());
			this->printDialog1 = (gcnew System::Windows::Forms::PrintDialog());
			this->comboBox1 = (gcnew System::Windows::Forms::ComboBox());
			this->textBox1 = (gcnew System::Windows::Forms::TextBox());
			this->menuStrip1->SuspendLayout();
			(cli::safe_cast<System::ComponentModel::ISupportInitialize^>(this->pictureBox1))->BeginInit();
			this->SuspendLayout();
			// 
			// menuStrip1
			// 
			this->menuStrip1->Items->AddRange(gcnew cli::array< System::Windows::Forms::ToolStripItem^  >(2) {
				this->fileToolStripMenuItem,
					this->aboutToolStripMenuItem
			});
			this->menuStrip1->Location = System::Drawing::Point(0, 0);
			this->menuStrip1->Name = L"menuStrip1";
			this->menuStrip1->Size = System::Drawing::Size(1051, 24);
			this->menuStrip1->TabIndex = 0;
			this->menuStrip1->Text = L"menuStrip1";
			// 
			// fileToolStripMenuItem
			// 
			this->fileToolStripMenuItem->DropDownItems->AddRange(gcnew cli::array< System::Windows::Forms::ToolStripItem^  >(4) {
				this->openToolStripMenuItem,
					this->saveToolStripMenuItem, this->printToolStripMenuItem, this->exitToolStripMenuItem
			});
			this->fileToolStripMenuItem->Name = L"fileToolStripMenuItem";
			this->fileToolStripMenuItem->Size = System::Drawing::Size(40, 20);
			this->fileToolStripMenuItem->Text = L"File";
			// 
			// openToolStripMenuItem
			// 
			this->openToolStripMenuItem->Name = L"openToolStripMenuItem";
			this->openToolStripMenuItem->Size = System::Drawing::Size(103, 22);
			this->openToolStripMenuItem->Text = L"Open";
			this->openToolStripMenuItem->Click += gcnew System::EventHandler(this, &MyForm::openToolStripMenuItem_Click);
			// 
			// saveToolStripMenuItem
			// 
			this->saveToolStripMenuItem->Name = L"saveToolStripMenuItem";
			this->saveToolStripMenuItem->Size = System::Drawing::Size(103, 22);
			this->saveToolStripMenuItem->Text = L"Save";
			this->saveToolStripMenuItem->Click += gcnew System::EventHandler(this, &MyForm::saveToolStripMenuItem_Click);
			// 
			// printToolStripMenuItem
			// 
			this->printToolStripMenuItem->Name = L"printToolStripMenuItem";
			this->printToolStripMenuItem->Size = System::Drawing::Size(103, 22);
			this->printToolStripMenuItem->Text = L"Print";
			this->printToolStripMenuItem->Click += gcnew System::EventHandler(this, &MyForm::printToolStripMenuItem_Click);
			// 
			// exitToolStripMenuItem
			// 
			this->exitToolStripMenuItem->Name = L"exitToolStripMenuItem";
			this->exitToolStripMenuItem->Size = System::Drawing::Size(103, 22);
			this->exitToolStripMenuItem->Text = L"Exit";
			this->exitToolStripMenuItem->Click += gcnew System::EventHandler(this, &MyForm::exitToolStripMenuItem_Click);
			// 
			// aboutToolStripMenuItem
			// 
			this->aboutToolStripMenuItem->Name = L"aboutToolStripMenuItem";
			this->aboutToolStripMenuItem->Size = System::Drawing::Size(58, 20);
			this->aboutToolStripMenuItem->Text = L"About";
			this->aboutToolStripMenuItem->Click += gcnew System::EventHandler(this, &MyForm::aboutToolStripMenuItem_Click);
			// 
			// checkbox1
			// 
			this->checkbox1->Location = System::Drawing::Point(43, 58);
			this->checkbox1->Name = L"checkbox1";
			this->checkbox1->Size = System::Drawing::Size(137, 23);
			this->checkbox1->TabIndex = 1;
			this->checkbox1->Text = L"Continuous sampling";
			this->checkbox1->UseVisualStyleBackColor = true;
//			this->checkbox1->Click += gcnew System::EventHandler(this, &MyForm::checkbox1_Click);
			// 
			// button1
			// 
			this->button1->Location = System::Drawing::Point(183, 58);
			this->button1->Name = L"button1";
			this->button1->Size = System::Drawing::Size(137, 23);
			this->button1->TabIndex = 1;
			this->button1->Text = L"Click to Start Sampling";
			this->button1->UseVisualStyleBackColor = true;
			this->button1->Click += gcnew System::EventHandler(this, &MyForm::button1_Click);
			// 
			// label1
			// 
			this->label1->AutoSize = true;
			this->label1->Location = System::Drawing::Point(343, 35);
			this->label1->Name = L"label1";
			this->label1->Size = System::Drawing::Size(56, 13);
			this->label1->TabIndex = 3;
			this->label1->Text = L"COM Port:";
			// 
			// comboBox2
			// 
			this->comboBox2->DropDownStyle = System::Windows::Forms::ComboBoxStyle::DropDownList;
			this->comboBox2->FormattingEnabled = true;
			this->comboBox2->Items->AddRange(gcnew cli::array< System::Object^  >(4) { L"14400", L"28800", L"57600", L"115200" });
			this->comboBox2->Location = System::Drawing::Point(604, 32);
			this->comboBox2->Name = L"comboBox2";
			this->comboBox2->Size = System::Drawing::Size(121, 21);
			this->comboBox2->TabIndex = 4;
			// 
			// label2
			// 
			this->label2->AutoSize = true;
			this->label2->Location = System::Drawing::Point(537, 35);
			this->label2->Name = L"label2";
			this->label2->Size = System::Drawing::Size(61, 13);
			this->label2->TabIndex = 5;
			this->label2->Text = L"Baud Rate:\r\n";
			// 
			// comboBox3
			// 
			this->comboBox3->DropDownStyle = System::Windows::Forms::ComboBoxStyle::DropDownList;
			this->comboBox3->FormattingEnabled = true;
			this->comboBox3->Items->AddRange(gcnew cli::array< System::Object^  >(4) { L"5", L"15", L"30", L"60" });
			this->comboBox3->Location = System::Drawing::Point(852, 32);
			this->comboBox3->Name = L"comboBox3";
			this->comboBox3->Size = System::Drawing::Size(136, 21);
			this->comboBox3->TabIndex = 6;
			// 
			// label3
			// 
			this->label3->AutoSize = true;
			this->label3->Location = System::Drawing::Point(741, 35);
			this->label3->Name = L"label3";
			this->label3->Size = System::Drawing::Size(105, 13);
			this->label3->TabIndex = 7;
			this->label3->Text = L"Sampling Time (sec):";
			// 
			// progressBar1
			// 
			this->progressBar1->BackColor = System::Drawing::SystemColors::Window;
			this->progressBar1->Location = System::Drawing::Point(446, 60);
			this->progressBar1->MarqueeAnimationSpeed = 1;
			this->progressBar1->Name = L"progressBar1";
			this->progressBar1->Size = System::Drawing::Size(380, 20);
			this->progressBar1->Step = 1;
			this->progressBar1->Style = System::Windows::Forms::ProgressBarStyle::Continuous;
			this->progressBar1->TabIndex = 8;
			// 
			// label4
			// 
			this->label4->AutoSize = true;
			this->label4->Location = System::Drawing::Point(343, 63);
			this->label4->Name = L"label4";
			this->label4->Size = System::Drawing::Size(97, 13);
			this->label4->TabIndex = 9;
			this->label4->Text = L"Sampling Progress:";
			// 
			// pictureBox1
			// 
			this->pictureBox1->BackColor = System::Drawing::SystemColors::Window;
			this->pictureBox1->BorderStyle = System::Windows::Forms::BorderStyle::FixedSingle;
			this->pictureBox1->Location = System::Drawing::Point(12, 90);
			this->pictureBox1->Name = L"pictureBox1";
			this->pictureBox1->Size = System::Drawing::Size(1027, 555);
			this->pictureBox1->TabIndex = 12;
			this->pictureBox1->TabStop = false;
			this->pictureBox1->Paint += gcnew System::Windows::Forms::PaintEventHandler(this, &EEGSampler::MyForm::pictureBox1_Paint);////ADDED
			// 
			// openFileDialog1
			// 
			this->openFileDialog1->FileName = L"openFileDialog1";
			// 
			// printDialog1
			// 
			this->printDialog1->UseEXDialog = true;
			// 
			// comboBox1
			// 
			this->comboBox1->DropDownStyle = System::Windows::Forms::ComboBoxStyle::DropDownList;
			this->comboBox1->FormattingEnabled = true;
			this->comboBox1->Location = System::Drawing::Point(405, 32);
			this->comboBox1->MaxDropDownItems = 15;
			this->comboBox1->Name = L"comboBox1";
			this->comboBox1->Size = System::Drawing::Size(121, 21);
			this->comboBox1->TabIndex = 13;
			// 
			// textBox1
			// 
			this->textBox1->BorderStyle = System::Windows::Forms::BorderStyle::FixedSingle;
			this->textBox1->Location = System::Drawing::Point(852, 60);
			this->textBox1->Name = L"textBox1";
			this->textBox1->Size = System::Drawing::Size(165, 20);
			this->textBox1->TabIndex = 14;
			// 
			// MyForm
			// 
			this->AutoScaleDimensions = System::Drawing::SizeF(6, 13);
			this->AutoScaleMode = System::Windows::Forms::AutoScaleMode::Font;
			this->BackColor = System::Drawing::Color::FromArgb(static_cast<System::Int32>(static_cast<System::Byte>(255)), static_cast<System::Int32>(static_cast<System::Byte>(224)),
				static_cast<System::Int32>(static_cast<System::Byte>(192)));
			this->ClientSize = System::Drawing::Size(1051, 657);
			this->Controls->Add(this->textBox1);
			this->Controls->Add(this->comboBox1);
			this->Controls->Add(this->pictureBox1);
			this->Controls->Add(this->label4);
			this->Controls->Add(this->progressBar1);
			this->Controls->Add(this->label3);
			this->Controls->Add(this->comboBox3);
			this->Controls->Add(this->label2);
			this->Controls->Add(this->comboBox2);
			this->Controls->Add(this->label1);
			this->Controls->Add(this->checkbox1);
			this->Controls->Add(this->button1);
			this->Controls->Add(this->menuStrip1);
			this->MainMenuStrip = this->menuStrip1;
			this->Name = L"MyForm";
			this->Text = L"Silicon Chip EEG Sampler";
			this->Load += gcnew System::EventHandler(this, &MyForm::MyForm_Load);
			this->menuStrip1->ResumeLayout(false);
			this->menuStrip1->PerformLayout();
			(cli::safe_cast<System::ComponentModel::ISupportInitialize^>(this->pictureBox1))->EndInit();
			this->ResumeLayout(false);
			this->PerformLayout();

			findPorts();
			Microsoft::Win32::RegistryKey^ key = Application::UserAppDataRegistry->OpenSubKey(this->Name);
			if( key ) {
				this->comboBox1->Text = key->GetValue(this->comboBox1->Name, this->comboBox1->Text)->ToString();
				this->comboBox2->Text = key->GetValue(this->comboBox2->Name, this->comboBox2->Text)->ToString();
				this->comboBox3->Text = key->GetValue(this->comboBox3->Name, this->comboBox3->Text)->ToString();
				this->checkbox1->Checked = Convert::ToBoolean(key->GetValue(this->checkbox1->Name, this->checkbox1->Checked ? "1" : "0")->ToString());
			}
			this->myTimer->Tick += gcnew EventHandler(this, &MyForm::TimerEventProcessor);
			this->myTimer->Interval = 1;
		}
#pragma endregion
//	---------------------------------------------------------------------------------------------------
// find available serial ports and make visible in comboBox1 list
	private: void findPorts(void)
	{
		array<Object^>^ objectArray = System::IO::Ports::SerialPort::GetPortNames();
		this->comboBox1->Items->AddRange(objectArray);
	}
//  ---------------------------------------------------------------------------------------------------
// "FILE>Open" file menu item click event
private: System::Void openToolStripMenuItem_Click(System::Object^ sender, System::EventArgs^  e)
{
	OpenFileDialog^ openFileDialog1 = gcnew OpenFileDialog;
	openFileDialog1->FileName = "MyEEGSample1";
	openFileDialog1->DefaultExt = "eeg";
	openFileDialog1->Filter = "EEG files (*.eeg) | *.eeg | All files(*.*) | *.*";
	DialogResult = openFileDialog1->ShowDialog();
	if (DialogResult == System::Windows::Forms::DialogResult::OK)	// result OK?
	{
		String^ filename = openFileDialog1->FileName;
		StreamReader^ sr = gcnew StreamReader(filename);
		this->textBox1->Text = sr->ReadLine();  // get the file's sampling date & time, -> textBox1 
		this->comboBox1->Text = sr->ReadLine();			// then the COM port used, -> comboBox1
		this->comboBox2->Text = sr->ReadLine();			// and the baud rate used, -> comboBox2
		intSamplingTime = Int16::Parse(sr->ReadLine()); // the sampling time
		intNumofSamples = Int16::Parse(sr->ReadLine()); // the number of samples
		// and finally the samples themselves
		for (int intCurrArrayCntr = 0; intCurrArrayCntr < intNumofSamples; intCurrArrayCntr++)
		{
			intSampleArray1[intCurrArrayCntr] = Int16::Parse(sr->ReadLine());
			intSampleArray2[intCurrArrayCntr] = Int16::Parse(sr->ReadLine());
			intSampleArray3[intCurrArrayCntr] = Int16::Parse(sr->ReadLine());
		}
		sr->Close();		// before closing the stream
		// now replace the text in comboBox3
		this->comboBox3->Text = intSamplingTime.ToString();
		if (!FormGraphic)
			FormGraphic = this->pictureBox1->CreateGraphics();

		// and draw graticules & plots in pictureBox1
		drawGratsandPlots(FormGraphic, Rectangle(0, 0, this->pictureBox1->Bounds.Width, this->pictureBox1->Bounds.Height));
	}
}	// end of FILE>Open File menu item click event
//  --------------------------------------------------------------------------------------------------
// "FILE>Save" file menu item click event
private: System::Void saveToolStripMenuItem_Click(System::Object^  sender, System::EventArgs^  e)
{
	SaveFileDialog^ saveFileDialog1 = gcnew SaveFileDialog;
	saveFileDialog1->CreatePrompt = true;
	saveFileDialog1->OverwritePrompt = true;
	saveFileDialog1->FileName = "MyEEGSample1";
	saveFileDialog1->DefaultExt = "eeg";
	saveFileDialog1->Filter = "EEG files (*.eeg)|*.eeg|All files (*.*)|*.*";
	DialogResult = saveFileDialog1->ShowDialog();	// show the saveFile dialog box
	if (DialogResult == System::Windows::Forms::DialogResult::OK)  // was result OK?
	{
		String^ filename = saveFileDialog1->FileName;
		StreamWriter^ sw = gcnew StreamWriter(filename);
		sw->WriteLine(this->textBox1->Text);			// save the date & time first
		sw->WriteLine(this->comboBox1->Text);			// then the COM port being used
		sw->WriteLine(this->comboBox2->Text);			// the baud rate used
		sw->WriteLine(intSamplingTime);					// the sampling time
		sw->WriteLine(intNumofSamples);					// the number of samples
		// and finally the samples themselves
		for (int intCurrArrayCntr = 0; intCurrArrayCntr < intNumofSamples; intCurrArrayCntr++)
		{
			sw->WriteLine(intSampleArray1[intCurrArrayCntr]);
			sw->WriteLine(intSampleArray2[intCurrArrayCntr]);
			sw->WriteLine(intSampleArray3[intCurrArrayCntr]);
		}
		sw->Close();		// before closing the file
		// then restore graticules & plots in pictureBox1
		drawGratsandPlots(FormGraphic, Rectangle(0, 0, this->pictureBox1->Bounds.Width, this->pictureBox1->Bounds.Height));
	}
}	// end of FILE>Save file menu item click event
//  ---------------------------------------------------------------------------------------------------
// "FILE>Print" file menu item click event
private: System::Void printToolStripMenuItem_Click(System::Object^  sender, System::EventArgs^  e)
{
	printDocument1->PrintPage += gcnew System::Drawing::Printing::PrintPageEventHandler(this, &EEGSampler::MyForm::PrintDocument1_PrintPage);
	// now set up the print dialog
	printDialog1->AllowSomePages = true;
	printDialog1->ShowHelp = true;
	printDialog1->AllowSelection = true;
	printDialog1->AllowCurrentPage = true;
	printDialog1->Document = printDocument1;
	printDialog1->PrinterSettings->DefaultPageSettings->Landscape = true;
	// now show the dialog box & allow user to select, etc.
	System::Windows::Forms::DialogResult result = printDialog1->ShowDialog();
	if (result == System::Windows::Forms::DialogResult::OK)
		printDocument1->Print();	// if result from dialog is OK, print document
}	// end of FILE>Print file menu item click event
// --------------------------------------------------------------------------------------------------
// "FILE>Exit" file menu item click event
private: System::Void exitToolStripMenuItem_Click(System::Object^  sender, System::EventArgs^  e)
{
	System::Windows::Forms::DialogResult buttonClicked;
	buttonClicked = MessageBox::Show("Sure you want to Exit?", "Confirm Exit", MessageBoxButtons::YesNo);
	if (buttonClicked == System::Windows::Forms::DialogResult::Yes)
		delete myRedPen;		// save system resources before leaving
		delete myBlackPen;
		delete FormGraphic;
		delete myBrush;
		this->Close();			// then close the form & depart
}	// end of FILE>Exit file menu click event
//	-----------------------------------------------------------------------------------------------
//	"ABOUT" menu item click event
private: System::Void aboutToolStripMenuItem_Click(System::Object^  sender, System::EventArgs^  e)
{
	String^message = "Silicon Chip EEG Sampler App V1.0" + "\nWritten by Jim Rowe, June 2018";
	MessageBox::Show(message, "Silicon Chip EEG Sampler", MessageBoxButtons::OK);
}
// ------------------------------------------------------------------------------------------------
// MyForm load event
private: System::Void MyForm_Load(System::Object^  sender, System::EventArgs^  e)
{
	DateTime localDate = DateTime::Now;
	this->textBox1->Text = localDate.ToString();	// display current date & time in textBox1
}
// -----------------------------------------------------------------------------------------------
// function to display the three graticules in pictureBox1 (each graticule 960 x 160)
private: void ShowGraticules(Graphics^ FormGraphic, Rectangle bounds)
{
	int gLeft = bounds.Left + 50;		// left side of graticules
	int gRight = bounds.Width - 17;		// right side of graticules
	int g1Top = bounds.Top + 10;		// top of Ch1 graticule
	int g1Bot = bounds.Top + 170;		// bottom of Ch1 graticule
	int g2Top = bounds.Top + 190;		// top of Ch2 graticule
	int g2Bot = bounds.Top + 350;		// bottom of Ch2 graticule
	int g3Top = bounds.Top + 370;		// top of Ch3 graticule
	int g3Bot = bounds.Top + 530;		// bottom of Ch3 graticule
	int gWidth = gRight - gLeft;		// width of the graticules (= 960)
	int gHeight = 160;					// height of each graticule

	// first wipe pictureBox1 in case we are replacing existing graticules & plots
	FormGraphic->FillRectangle(myBrush, bounds/*Rectangle(0,0,1027,553)*/);	// myBrush is white initially
	// then make sure we'll be using a black pen
	if (!myBlackPen)
		myBlackPen = gcnew System::Drawing::Pen(System::Drawing::Color::Black);	// set pen colour
	myBlackPen->Width = 2.0F;	// & its width to 2 pixels, for bolder axis lines
	// then draw the vertical axis lines
	FormGraphic->DrawLine(myBlackPen, gLeft, g1Top, gLeft, g1Bot);
	FormGraphic->DrawLine(myBlackPen, gLeft, g2Top, gLeft, g2Bot);
	FormGraphic->DrawLine(myBlackPen, gLeft, g3Top, gLeft, g3Bot);
	// & the centre time axis lines
	FormGraphic->DrawLine(myBlackPen, gLeft, (g1Top + 80), gRight, (g1Top + 80));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g2Top + 80), gRight, (g2Top + 80));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g3Top + 80), gRight, (g3Top + 80));
	// now reset myBlackPen for drawing the lighter lines
	myBlackPen->Width = 1.0F;
	// and draw the three graticule boxes
	FormGraphic->DrawRectangle(myBlackPen, Rectangle(gLeft, g1Top, gWidth, gHeight));
	FormGraphic->DrawRectangle(myBlackPen, Rectangle(gLeft, g2Top, gWidth, gHeight));
	FormGraphic->DrawRectangle(myBlackPen, Rectangle(gLeft, g3Top, gWidth, gHeight));
	// then draw the horizontal signal level lines in each graticule
	FormGraphic->DrawLine(myBlackPen, gLeft, (g1Top + 20), gRight, (g1Top + 20));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g1Top + 40), gRight, (g1Top + 40));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g1Top + 60), gRight, (g1Top + 60));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g1Top + 100), gRight, (g1Top + 100));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g1Top + 120), gRight, (g1Top + 120));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g1Top + 140), gRight, (g1Top + 140));

	FormGraphic->DrawLine(myBlackPen, gLeft, (g2Top + 20), gRight, (g2Top + 20));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g2Top + 40), gRight, (g2Top + 40));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g2Top + 60), gRight, (g2Top + 60));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g2Top + 100), gRight, (g2Top + 100));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g2Top + 120), gRight, (g2Top + 120));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g2Top + 140), gRight, (g2Top + 140));

	FormGraphic->DrawLine(myBlackPen, gLeft, (g3Top + 20), gRight, (g3Top + 20));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g3Top + 40), gRight, (g3Top + 40));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g3Top + 60), gRight, (g3Top + 60));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g3Top + 100), gRight, (g3Top + 100));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g3Top + 120), gRight, (g3Top + 120));
	FormGraphic->DrawLine(myBlackPen, gLeft, (g3Top + 140), gRight, (g3Top + 140));

	// now draw the vertical time unit lines in each graticule
	if (intSamplingTime) {////ADDED
		intSecSp = 960 / intSamplingTime;	// work out the seconds interline spacing
		int LCtr = 1;						// and set up a line counter
		for (LCtr = 1; (LCtr * intSecSp) < 960; LCtr++)
		{
			FormGraphic->DrawLine(myBlackPen, (gLeft + (LCtr * intSecSp)), g1Top, (gLeft + (LCtr * intSecSp)), (g1Top + 160));
			FormGraphic->DrawLine(myBlackPen, (gLeft + (LCtr * intSecSp)), g2Top, (gLeft + (LCtr * intSecSp)), (g2Top + 160));
			FormGraphic->DrawLine(myBlackPen, (gLeft + (LCtr * intSecSp)), g3Top, (gLeft + (LCtr * intSecSp)), (g3Top + 160));
		}
	}////ADDED
	
	// next draw the various text labels on the graticules
	System::Drawing::Font^drawFont = gcnew System::Drawing::Font("Arial", 10);	// set the font & size
	System::Drawing::SolidBrush^ drawBrush = gcnew System::Drawing::SolidBrush(System::Drawing::Color::Black);	// brush colour
	System::Drawing::StringFormat^ drawFormat = gcnew System::Drawing::StringFormat();	// & the format

	String^ drawString = "SECONDS";
	float x = (float)(gLeft + (gWidth * 3/4));	// define its X location
	float y = (float)(g3Bot + 8);			// its Y location
	drawFormat->Alignment = StringAlignment::Center;	// & its alignment
	FormGraphic->DrawString(drawString, drawFont, drawBrush, x, y, drawFormat);	// then draw it

	// then draw the time units
	drawFont = gcnew System::Drawing::Font("Arial", 8);	// first reduce the font size
	Int16 intUmult = 5;		// define multiplier for unit labels
	switch (intSamplingTime)
	{
	case 5:
		intUmult = 1;
		break;
	case 15:
		intUmult = 5;
		break;
	case 30:
		intUmult = 5;
		break;
	case 60:
		intUmult = 10;
		break;
	}
	Int16 intUCtr = 0;		// define a loop counter for time unit labels
	for (intUCtr = 0; (intUCtr * intUmult) <= intSamplingTime; intUCtr++)
	{
		drawString = (intUCtr * intUmult).ToString();	// make string from intUmult * intUCtr
		drawFormat->Alignment = StringAlignment::Center;	// & set for centre alignment
		x = (float)(gLeft + (intUCtr * intUmult * intSecSp));	// work out X position
		FormGraphic->DrawString(drawString, drawFont, drawBrush, x, (float)(g1Bot + 2), drawFormat);		// draw for grat 1
		FormGraphic->DrawString(drawString, drawFont, drawBrush, x, (float)(g2Bot + 2), drawFormat);	// & for grat 2
		FormGraphic->DrawString(drawString, drawFont, drawBrush, x, (float)(g3Bot + 2), drawFormat);		// & for grat 3
	}
	// & finally draw the Channel legends
	drawFormat->Alignment = StringAlignment::Center;	// set for centre X alignment 
	drawFormat->LineAlignment = StringAlignment::Center;	// & centre Y alignment
	drawString = "Ch 1";
	FormGraphic->DrawString(drawString, drawFont, drawBrush, (float)(gLeft - 20), (float)(g1Bot - 80), drawFormat);	// draw Ch1 legend
	drawString = "Ch 2";
	FormGraphic->DrawString(drawString, drawFont, drawBrush, (float)(gLeft - 20), (float)(g2Bot - 80), drawFormat);	// & Ch2 legend
	drawString = "Ch 3";
	FormGraphic->DrawString(drawString, drawFont, drawBrush, (float)(gLeft - 20), (float)(g3Bot - 80), drawFormat);	// & Ch3 legend
} // end of ShowGraticules function
// -----------------------------------------------------------------------------------------------
// "Start/Stop Sampling" button click event
private: System::Void button1_Click(System::Object^  sender, System::EventArgs^  e)
{
	// first check that user doesn't want to stop sampling
	if (this->samplingActive == true)
	{
		StopSampling();
#ifdef ENABLE_SERIAL
		this->serialPort1->WriteLine("s");	// send Stop Sampling command char to EEG Sampler
		this->serialPort1->Close();			// and close port
#endif
	}
	else
	{
		this->samplingActive = true;	// must want to start sampling, so do so

		// but make sure all three comboBoxes have been set up, & none is empty
		if (this->comboBox1->Text == String::Empty || this->comboBox2->Text == String::Empty || this->comboBox3->Text == String::Empty)
		{
			MessageBox::Show("Please Select Port, Baud Rate & Sampling Time!", "Settings Not Made", MessageBoxButtons::OK);
			return;
		}
		else
		{
			Microsoft::Win32::RegistryKey^ key = Application::UserAppDataRegistry->CreateSubKey(this->Name);
			if (key) {
				key->SetValue(this->comboBox1->Name, this->comboBox1->Text);
				key->SetValue(this->comboBox2->Name, this->comboBox2->Text);
				key->SetValue(this->comboBox3->Name, this->comboBox3->Text);
				key->SetValue(this->checkbox1->Name, this->checkbox1->Checked);
			}
#ifdef ENABLE_SERIAL
			// port settings must have been made, so we can proceed to open it
			try
			{
				// but don't do this if it's already open
				if (!this->serialPort1->IsOpen)
				{
					// set the port's PortName property
					this->serialPort1->PortName = this->comboBox1->Text;
					// then set its BaudRate property
					this->serialPort1->BaudRate = Int32::Parse(this->comboBox2->Text);
					// and open the serial port
					this->serialPort1->Open();
				}
			}
			catch (UnauthorizedAccessException^)
			{
				MessageBox::Show("Unauthorised Access", "Unauthorised Access", MessageBoxButtons::OK);
			}
#endif
		}	// end of port opening sequence
			// now check if Stop button was clicked
		if (this->samplingActive == false)
		{
			this->serialPort1->WriteLine("s");	// send Stop Sampling command char to EEG Sampler
			this->serialPort1->Close();			// and close port
			return;								// then exit
		}

		// next get selected sampling time period from comboBox3
		this->intSamplingTime = Int16::Parse(this->comboBox3->Text);
		// and work out number of triple samples to be taken
		intNumofSamples = intSmplPerSec * (this->intSamplingTime);	// will be 1335, 4005, 8010 or 16020

		// then get the current date & time, to display in textBox1
		DateTime localDate = DateTime::Now;
		this->textBox1->Text = localDate.ToString();

		// now setup for drawing graphics in pictureBox1
		if (!FormGraphic)
			FormGraphic = this->pictureBox1->CreateGraphics();
		// then go & display the graticules
		ShowGraticules(FormGraphic, Rectangle(0, 0, this->pictureBox1->Bounds.Width, this->pictureBox1->Bounds.Height));

		// now change text in button1, to allow user to stop sampling
		this->button1->Text = "Click to Stop Sampling";

		// also initialise the progress bar
		this->progressBar1->Visible = true;				// make it visible
		this->progressBar1->Minimum = 1;				// set minimum
		this->progressBar1->Maximum = intNumofSamples;	// and maximum
		this->progressBar1->Step = 1;	// set the step size
		this->progressBar1->Value = 1;	// and the bar's initial value

		// next disable the toolstrip menu items, to prevent possible conflicts during sampling
		this->openToolStripMenuItem->Enabled = false;
		this->saveToolStripMenuItem->Enabled = false;
		this->printToolStripMenuItem->Enabled = false;
		this->exitToolStripMenuItem->Enabled = false;

		// now do the actual sampling 
		this->samplingActive = true;
		this->samplingLastShownPoint = 0;	// reset samplingLastShownPoint

		intCurrArrayCntr = 0;
#ifdef ENABLE_SERIAL
		this->serialPort1->WriteLine("t");	// send sampling command char to EEG Sampler
#endif
		this->myTimer->Start();
#if 0
		// start of actual sampling loop
		for (intCurrArrayCntr = 0; this->samplingActive && intCurrArrayCntr < intNumofSamples; intCurrArrayCntr++)
		{
#ifdef ENABLE_SERIAL
			this->serialPort1->WriteLine("t");	// send sampling command char to EEG Sampler
#endif
			try
			{
#ifdef ENABLE_SERIAL
				intSampleArray1[intCurrArrayCntr] = Int16::Parse(this->serialPort1->ReadLine()); // read 1st line & save in Array1
				intSampleArray2[intCurrArrayCntr] = Int16::Parse(this->serialPort1->ReadLine()); // read 2nd line & save in Array2
				intSampleArray3[intCurrArrayCntr] = Int16::Parse(this->serialPort1->ReadLine()); // read 3rd line & save in Array3
			}
#endif
			catch (TimeoutException^)
			{
				timeout = true;
				break;
			}
			// then go and update the progress bar and the three plots
			UpdatePlots(FormGraphic, Rectangle(0, 0, this->pictureBox1->Bounds.Width, this->pictureBox1->Bounds.Height));

		}	// now loop back to get next trio of samples

		// once sampling is done, send 'stop sampling' char to EEG Sampler & close serialPort
#ifdef ENABLE_SERIAL
		this->serialPort1->WriteLine("s");	// send stop sampling command char
		this->serialPort1->Close();			// and close port
#endif
		if (timeout)
			MessageBox::Show("Port not responding", "Timeout exception", MessageBoxButtons::OK);
		// then reset progress bar
		this->progressBar1->Value = 1;		// return to the bar's initial value
		// restore the button1 legend to 'Start Sampling'
		this->button1->Text = "Click to Start Sampling";
		this->samplingActive = false;////ADDED

		// & re-enable the toolstrip menu items before leaving
		this->openToolStripMenuItem->Enabled = true;
		this->saveToolStripMenuItem->Enabled = true;
		this->printToolStripMenuItem->Enabled = true;
		this->exitToolStripMenuItem->Enabled = true;
#endif
	}
}	// end of "Start Sampling" button click event

// -----------------------------------------------------------------------------------------------
// function to update the three plots, also the progress bar
private: System::Void UpdatePlots(Graphics^ FormGraphic, Rectangle bounds)
{
	this->progressBar1->PerformStep();
	
	float lineWidth = 2.0F;		// set the width of the plots
	// then plot a new point in each graticule
	PlotCh1Sample(FormGraphic, bounds, lineWidth, intCurrArrayCntr);
	PlotCh2Sample(FormGraphic, bounds, lineWidth, intCurrArrayCntr);
	PlotCh3Sample(FormGraphic, bounds, lineWidth, intCurrArrayCntr);
	this->samplingLastShownPoint = intCurrArrayCntr;
}
// -----------------------------------------------------------------------------------------------
// function to plot current Ch1 sample as a line between this sample and the previous one
		 // Note: samplePos = intCurArrayCntr
private: void PlotCh1Sample(Graphics^ FormGraphic, Rectangle bounds, float lineWidth, int samplePos)
{
	if (!myRedPen)
		myRedPen = gcnew System::Drawing::Pen(System::Drawing::Color::Red);
	myRedPen->Width = lineWidth;
	if (samplePos == 0)
		point1 = getSmplC1XY(bounds, 0, intSampleArray1[samplePos]);	// make point1 start at centre of graticule
	else
		point1 = getSmplC1XY(bounds, (samplePos - 1), intSampleArray1[samplePos - 1]); // get line start
	point2 = getSmplC1XY(bounds, samplePos, intSampleArray1[samplePos]);	 // get line finish
	FormGraphic->DrawLine(myRedPen, point1, point2);	// then draw the line
}	// end of plot the current sample for channel 1 function
// ------------------------------------------------------------------------------------------------
// function to get the current Ch1 sample Point X & Y co-ordinates 
private: Point getSmplC1XY(Rectangle bounds, int SamplePos, int rawSampleValue)
{
	int gLeft = bounds.Left + 50;		// left side of graticules
	int gRight = bounds.Width - 17;		// right side of graticules
	int g1Bot = bounds.Top + 170;		// bottom of Ch1 graticule
	int gWidth = gRight - gLeft;		// width of the graticules (= 960)
	int gHeight = 160;					// height of each graticule

	int intScaledSmplVal = (int)(rawSampleValue / 6.4); // scale sample back to <0 - 160>
	if (intScaledSmplVal > 160)
		intScaledSmplVal = 160;		// clamp the maximum to 160
	else if (intScaledSmplVal < 0)
		intScaledSmplVal = 0;		// and the minimum to 0

	// now get new point's X position
	float x = (float)(gLeft + (gWidth * SamplePos / intNumofSamples));
	// and its Y position
	float y = (float)(g1Bot - intScaledSmplVal);
	return Point((int)x, (int)y);
}	// end of getSmplC1XY function
// ------------------------------------------------------------------------------------------------
// function to plot current Ch2 sample as a line between this sample and the previous one
private: void PlotCh2Sample(Graphics^ FormGraphic, Rectangle bounds, float lineWidth, int samplePos)
{
	if (!myRedPen)
		myRedPen = gcnew System::Drawing::Pen(System::Drawing::Color::Red);
	myRedPen->Width = lineWidth;
	if (samplePos == 0)
		point1 = getSmplC2XY(bounds, 0, intSampleArray2[samplePos]);	// make point1 start centre of graticule
	else
		point1 = getSmplC2XY(bounds, (samplePos - 1), intSampleArray2[samplePos - 1]); // get line start
	point2 = getSmplC2XY(bounds, samplePos, intSampleArray2[samplePos]);	 // get line finish
	FormGraphic->DrawLine(myRedPen, point1, point2);	// then draw the line
}	// end of plot the current sample for channel 1 function
// ------------------------------------------------------------------------------------------------
// function to get the current Ch2 sample Point X & Y co-ordinates 
private: Point getSmplC2XY(Rectangle bounds, int SamplePos, int rawSampleValue)
{
	int gLeft = bounds.Left + 50;		// left side of graticules
	int gRight = bounds.Width - 17;		// right side of graticules
	int g2Bot = bounds.Top + 350;		// bottom of Ch2 graticule
	int gWidth = gRight - gLeft;		// width of the graticules (= 960)
	int gHeight = 160;					// height of each graticule

	int intScaledSmplVal = (int)(rawSampleValue / 6.4); // scale sample back to <0 - 160>
	if (intScaledSmplVal > 160)
		intScaledSmplVal = 160;		// clamp the maximum to 160
	else if (intScaledSmplVal < 0)
		intScaledSmplVal = 0;		// and the minimum to 0

	// now get new point's X position
	float x = (float)(gLeft + (gWidth * SamplePos / intNumofSamples));
	// and its Y position
	float y = (float)(g2Bot - intScaledSmplVal);
	return Point((int)x, (int)y);
}	// end of getSmplC2XY function
// --------------------------------------------------------------------------------------------
// function to plot current Ch3 sample as a line between this sample and the previous one
private: void PlotCh3Sample(Graphics^ FormGraphic, Rectangle bounds, float lineWidth, int samplePos)
{
	if (!myRedPen)
		myRedPen = gcnew System::Drawing::Pen(System::Drawing::Color::Red);
	myRedPen->Width = lineWidth;
	if (samplePos == 0)
		point1 = getSmplC3XY(bounds, 0, intSampleArray3[samplePos]);	// make point1 start centre of graticule
	else
		point1 = getSmplC3XY(bounds, (samplePos - 1), intSampleArray3[samplePos - 1]); // get line start
	point2 = getSmplC3XY(bounds, samplePos, intSampleArray3[samplePos]);	 // get line finish
	FormGraphic->DrawLine(myRedPen, point1, point2);	// then draw the line
}	// end of plot the current sample for channel 3 function
// ------------------------------------------------------------------------------------------------
// function to get the current Ch3 sample Point X & Y co-ordinates 
private: Point getSmplC3XY(Rectangle bounds, int SamplePos, int rawSampleValue)
{
	int gLeft = bounds.Left + 50;		// left side of graticules
	int gRight = bounds.Width - 17;		// right side of graticules
	int g3Bot = bounds.Top + 530;		// bottom of Ch3 graticule
	int gWidth = gRight - gLeft;		// width of the graticules (= 960)
	int gHeight = 160;					// height of each graticule

	int intScaledSmplVal = (int)(rawSampleValue / 6.4); // scale sample back to <0 - 160>
	if (intScaledSmplVal > 160)
		intScaledSmplVal = 160;		// clamp the maximum to 160
	else if (intScaledSmplVal < 0)
		intScaledSmplVal = 0;		// and the minimum to 0

	// now get new point's X position
	float x = (float)(gLeft + (gWidth * SamplePos / intNumofSamples));
	// and its Y position
	float y = (float)(g3Bot - intScaledSmplVal);
	return Point((int)x, (int)y);
}	// end of getSmplC3XY function
// --------------------------------------------------------------------------------------------
// function to create image for printing EEG plots
private: System::Void PrintDocument1_PrintPage(System::Object ^sender, System::Drawing::Printing::PrintPageEventArgs ^e)
{
	Rectangle bounds = e->PageBounds;
	// create some blank space in the margins
	bounds.Width -= 40;
	bounds.Height -= 40;
	bounds.X += 20;
	bounds.Y += 20;
	// reserve some space for a title & extra info at the top
	Rectangle textBounds = bounds;
	textBounds.Height = 60;
	bounds.Y = bounds.Y + 60;
	bounds.Height = bounds.Height - 60;
	// now draw that text (function below)
	drawPrintoutText(e->Graphics, textBounds);
	// ensure graphs are not too tall
	int max_bottom = bounds.Top + bounds.Width * 9 / 16;
	if (bounds.Bottom > max_bottom)
		bounds.Height = max_bottom - bounds.Top;
	RedrawGraticuleandPlots(e->Graphics, bounds, 1.0f);
}	// end of PrintDocument1_PrintPage function
// --------------------------------------------------------------------------------------------
// function to redraw graticule and plots
private: void RedrawGraticuleandPlots(Graphics^ pGraphics, Rectangle bounds, float lineWidth)
{
	ShowGraticules(pGraphics, bounds);
	intNumofSamples = this->intNumofSamples;
	for (int intCurrArrayCntr = 0; intCurrArrayCntr < intNumofSamples; intCurrArrayCntr++)
	{
		PlotCh1Sample(pGraphics, bounds, lineWidth, /*intSampleArray1[*/intCurrArrayCntr/*]*/);////CHANGED
		PlotCh2Sample(pGraphics, bounds, lineWidth, /*intSampleArray2[*/intCurrArrayCntr/*]*/);////CHANGED
		PlotCh3Sample(pGraphics, bounds, lineWidth, /*intSampleArray3[*/intCurrArrayCntr/*]*/);////CHANGED
	}
}	// end of function to redraw graticule and plots
// ---------------------------------------------------------------------------------------------
// function to add text to printout of plots
private: System::Void drawPrintoutText(Graphics^ pGraphics, Rectangle bounds)
{
	// first set the font & size for the title and the data text
	System::Drawing::Font^titleFont = gcnew System::Drawing::Font("Arial Bold", 14);
	System::Drawing::Font^textFont = gcnew System::Drawing::Font("Arial", 10);
	// also the brush size and colour, and the string format
	System::Drawing::SolidBrush^ drawBrush = gcnew System::Drawing::SolidBrush(System::Drawing::Color::Black);
	System::Drawing::StringFormat^ drawFormat = gcnew System::Drawing::StringFormat();

	// then print out the title plus the date and time
	String^ drawString = "SILICON CHIP EEG PLOT, " + this->textBox1->Text;
	float x = (float)(bounds.Left + bounds.Right) / 2.0f;
	float y = (float)bounds.Top + 10;
	drawFormat->Alignment = StringAlignment::Center;
	pGraphics->DrawString(drawString, titleFont, drawBrush, x, y, drawFormat);

	// and finally assemble & print out the data text line
	drawString = "Baud Rate: " + this->comboBox2->Text + "   COM Port: " + this->comboBox1->Text;
	drawString = drawString + ",   Sampling Time: " + intSamplingTime + " seconds";
	x = (float)(bounds.Left + bounds.Right) / 2.0f;
	y = (float)bounds.Top + 40;
	drawFormat->Alignment = StringAlignment::Center;
	pGraphics->DrawString(drawString, textFont, drawBrush, x, y, drawFormat);
}	// end of drawPrintoutText function
// --------------------------------------------------------------------------------------------
private: System::Void pictureBox1_Paint(System::Object^ sender, System::Windows::Forms::PaintEventArgs^ e)
{
	RedrawGraticuleandPlots(e->Graphics, Rectangle(0, 0, this->pictureBox1->Bounds.Width, this->pictureBox1->Bounds.Height), 2.0f);
}
	protected: virtual void OnResize(EventArgs^ e) override
	{
		Rectangle bounds = this->Bounds;
		this->pictureBox1->Size = System::Drawing::Size(1027, 553);
		this->pictureBox1->Invalidate();
	}
// --------------------------------------------------------------------------------------------
	// function to draw graticules and plots
	private: System::Void drawGratsandPlots(Graphics^ pGraphics, Rectangle bounds)
	{
		//this->pictureBox1->Invalidate();
		// first display the graticules
		ShowGraticules(pGraphics, Rectangle(0, 0, this->pictureBox1->Bounds.Width, this->pictureBox1->Bounds.Height));
		// then display the three plots
		float lineWidth = 2.5F;		// set the width of the plots
		intNumofSamples = this->intNumofSamples;
		for (int intCurrArrayCntr = 0; intCurrArrayCntr < intNumofSamples; intCurrArrayCntr++)
		{
			PlotCh1Sample(pGraphics, bounds, lineWidth, intSampleArray1[intCurrArrayCntr]);
			PlotCh2Sample(pGraphics, bounds, lineWidth, intSampleArray2[intCurrArrayCntr]);
			PlotCh3Sample(pGraphics, bounds, lineWidth, intSampleArray3[intCurrArrayCntr]);
		}
	}
// ----------------------------------------------------------------------------------------------
};	// end of public re class MyForm : public System::Windows::Forms:: MyForm
}	// end of EEGSampler namespace
